// Fill out your copyright notice in the Description page of Project Settings.


#include "PlayerCharacter.h"

/* Components */
#include "Camera/CameraComponent.h"
#include "GameFramework/SpringArmComponent.h"

#include "GameFramework/CharacterMovementComponent.h"
#include "GameFramework/Controller.h"
 
/* Input */
#include "Components/InputComponent.h"
#include "EnhancedInputComponent.h"
#include "EnhancedInputSubsystems.h"

#include "Math/UnrealMathUtility.h"
#include "Kismet/KismetMathLibrary.h"

/* Includes to track Hit and Overlap collisions */
// #include "Obstacle.h"
// #include "EnemyProjectile.h"
#include "EnemyMeleeCharacter.h"
#include "Weapon.h"
#include "AmmoCrate.h"
// #include "EnemyRangeCharacter.h"

// Sets default values
APlayerCharacter::APlayerCharacter()
{
 	// Set this character to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	CapsuleComponent = GetCapsuleComponent();

	/* Capsule size */
	CapsuleComponent->InitCapsuleSize(30.f, 100.f);

	CapsuleComponent->SetSimulatePhysics(false);

	
	/* Don't rotate when the controller rotates */
	bUseControllerRotationPitch = false;
	bUseControllerRotationYaw = false;
	bUseControllerRotationRoll = false;

	/* Character movement */
	GetCharacterMovement()->bOrientRotationToMovement = true; // Character moves in the direction of input	
	GetCharacterMovement()->RotationRate = FRotator(0.0f, 500.0f, 0.0f); // Rotation rate

	GetCharacterMovement()->MaxWalkSpeed = 500.f;
	GetCharacterMovement()->MinAnalogWalkSpeed = 20.f;
	GetCharacterMovement()->BrakingDecelerationWalking = 2000.f;

	// Create a camera boom (pulls in towards the player if there is a collision)
	CameraBoom = CreateDefaultSubobject<USpringArmComponent>(TEXT("CameraBoom"));
	CameraBoom->SetupAttachment(CapsuleComponent);
	CameraBoom->TargetArmLength = 1000.0f; // The camera follows at this distance behind the character	
	CameraBoom->bUsePawnControlRotation = false; // Static camera
	CameraBoom->SetRelativeRotation(FRotator(-40, 0, 0));

	// Create a follow camera
	FollowCamera = CreateDefaultSubobject<UCameraComponent>(TEXT("FollowCamera"));
	FollowCamera->SetupAttachment(CameraBoom, USpringArmComponent::SocketName); // Attach the camera to the end of the boom
	FollowCamera->bUsePawnControlRotation = false;
	
	//static ConstructorHelpers::FObjectFinder<UAnimSequence> Anim_Walk(TEXT("AnimSequence'/Game/Mannequin/Animations/ThirdPersonJump_Start.ThirdPersonJump_Start'"));
	//Anim = Anim_Walk.Object;
}

// Called when the game starts or when spawned
void APlayerCharacter::BeginPlay()
{
	Super::BeginPlay();

	/* Setting up Hit/Overlap events for player's capsule collider */
	CapsuleComponent->OnComponentBeginOverlap.AddDynamic(this, &APlayerCharacter::OnOverlapBegin);
	CapsuleComponent->OnComponentEndOverlap.AddDynamic(this, &APlayerCharacter::OnOverlapEnd);
	CapsuleComponent->OnComponentHit.AddDynamic(this, &APlayerCharacter::OnHit);

	if (APlayerController* PlayerController = Cast<APlayerController>(Controller))
	{
		if (UEnhancedInputLocalPlayerSubsystem* Subsystem = ULocalPlayer::GetSubsystem<UEnhancedInputLocalPlayerSubsystem>(PlayerController->GetLocalPlayer()))
		{
			Subsystem->AddMappingContext(PlayerContext, 0);
		}

		PlayerController->SetShowMouseCursor(false); // Shows the cursor
		PlayerController->bEnableClickEvents = true;         // Enables click events
		PlayerController->bEnableMouseOverEvents = true;     // Enables mouse-over events

		//FInputModeGameAndUI InputMode;
		//PlayerController->SetInputMode(InputMode);
	}

	/* Resetting this up for future restarts */
	bCanMove = true;
	bCanDodge = true;
	PlayerHealth = MaxPlayerHealth;

	EWhenTakesDamage();
	EWhenAttacks();

	ProjectileSpawnComponent = Cast<UStaticMeshComponent>(GetDefaultSubobjectByName(TEXT("ProjectileSpawnLocation")));

	WeaponBackpack_Axe[0] = Cast<UChildActorComponent>(GetDefaultSubobjectByName(TEXT("Weapon_Axe1")));
	WeaponBackpack_Axe[1] = Cast<UChildActorComponent>(GetDefaultSubobjectByName(TEXT("Weapon_Axe2")));
	WeaponBackpack_Axe[2] = Cast<UChildActorComponent>(GetDefaultSubobjectByName(TEXT("Weapon_Axe3")));

	if (WidgetsClass)
	{
		UI_Interact = CreateWidget<UInteract>(GetWorld(), WidgetsClass);
		if (UI_Interact)
		{
			UI_Interact->AddToViewport();
			SetInteractVisibility(false);
		}
	}

	
}



// Called every frame
void APlayerCharacter::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
	UpdateRotationTowardsCursor(CapsuleComponent,DeltaTime);



	/* Axe "backpack" visualization */
	if (AxeAmmoInMag == 0)
	{
		WeaponBackpack_Axe[0]->SetVisibility(false);
		WeaponBackpack_Axe[1]->SetVisibility(false);
		WeaponBackpack_Axe[2]->SetVisibility(false);
	}
	else if (AxeAmmoInMag == 1)
	{
		WeaponBackpack_Axe[0]->SetVisibility(true);
		WeaponBackpack_Axe[1]->SetVisibility(false);
		WeaponBackpack_Axe[2]->SetVisibility(false);
	}
	else if (AxeAmmoInMag == 2)
	{
		WeaponBackpack_Axe[0]->SetVisibility(true);
		WeaponBackpack_Axe[1]->SetVisibility(true);
		WeaponBackpack_Axe[2]->SetVisibility(false);
	}
	else if (AxeAmmoInMag == 3)
	{
		WeaponBackpack_Axe[0]->SetVisibility(true);
		WeaponBackpack_Axe[1]->SetVisibility(true);
		WeaponBackpack_Axe[2]->SetVisibility(true);
	}

	/* Ends game if health drops to/below 0 */
	if (PlayerHealth <= 0)
		GameOver();
}

/* Player's inputs. Used EnhancedInput library */
void APlayerCharacter::SetupPlayerInputComponent(UInputComponent* PlayerInputComponent)
{
	Super::SetupPlayerInputComponent(PlayerInputComponent);

	if (UEnhancedInputComponent* EnhancedInputComponent = CastChecked<UEnhancedInputComponent>(PlayerInputComponent))
	{
		EnhancedInputComponent->BindAction(MovementAction, ETriggerEvent::Triggered, this, &APlayerCharacter::Move);
		EnhancedInputComponent->BindAction(AttackAction, ETriggerEvent::Triggered, this, &APlayerCharacter::Attack);
		EnhancedInputComponent->BindAction(DodgeAction, ETriggerEvent::Triggered, this, &APlayerCharacter::Dodge);
		EnhancedInputComponent->BindAction(InteractAction, ETriggerEvent::Triggered, this, &APlayerCharacter::Interact);
		EnhancedInputComponent->BindAction(ReloadAction, ETriggerEvent::Triggered, this, &APlayerCharacter::Reload);
		EnhancedInputComponent->BindAction(AttackAlternativeAction, ETriggerEvent::Triggered, this, &APlayerCharacter::AttackAlternative);
		EnhancedInputComponent->BindAction(ItemSlot1Action, ETriggerEvent::Triggered, this, &APlayerCharacter::ItemSlot1);
		EnhancedInputComponent->BindAction(ItemSlot2Action, ETriggerEvent::Triggered, this, &APlayerCharacter::ItemSlot2);
		EnhancedInputComponent->BindAction(RotateCamera_LeftAction, ETriggerEvent::Triggered, this, &APlayerCharacter::RotateCamera_Left);
		EnhancedInputComponent->BindAction(RotateCamera_RightAction, ETriggerEvent::Triggered, this, &APlayerCharacter::RotateCamera_Right);
	}

}

/* Player abilities */
void APlayerCharacter::Move(const FInputActionValue& Value)
{
	if (bCanMove)
	{
		const FVector2D MovementVector = Value.Get<FVector2D>();

		const FRotator Rotation = CameraBoom->GetRelativeRotation();
		const FRotator YawRotation(0.f, Rotation.Yaw, 0.f);

		const FVector ForwardDirection = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::X);
		const FVector RightDirection = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::Y);

		AddMovementInput(ForwardDirection, MovementVector.Y);
		AddMovementInput(RightDirection, MovementVector.X);
	}
	
	/* Second way 
	const FVector Forward = GetActorForwardVector();
	const FVector Right = GetActorRightVector();
	AddMovementInput(Forward, MovementVector.Y);
	AddMovementInput(Right, MovementVector.X);
	*/
}

/* Combat */
void APlayerCharacter::Attack()
{
	if (bCanAttack)
	{
		float Ammo = 0;
		if (PlayerCurrentWeapon == gun)
			Ammo = PrimaryAmmoInMag;
		else if (PlayerCurrentWeapon == axe)
			Ammo = AxeAmmoInMag;


		if (Ammo > 0)
		{
			bCanAttack = false;

			FTransform SpawnLocationTransform = ProjectileSpawnComponent->GetComponentTransform();

			FActorSpawnParameters SpawnInfo = FActorSpawnParameters();

			if (PlayerCurrentWeapon == gun)
			{
				GetWorld()->SpawnActor<APlayerProjectile>(Projectile, SpawnLocationTransform, SpawnInfo);
				PrimaryAmmoInMag--;
			}
			else if (PlayerCurrentWeapon == axe)
			{
				AWeapon* NewAxe = GetWorld()->SpawnActor<AWeapon>(Axe, SpawnLocationTransform, SpawnInfo);
				NewAxe->bIsThrown = true;
				NewAxe->bIsStatic = false;
				AxeAmmoInMag--;
			}

			EWhenAttacks();

			GetWorldTimerManager().SetTimer(AttackDelayTimerHandle, this, &APlayerCharacter::ResetAttack, AttackDelay, false);

			
		}
		else
		{
			if (PlayerCurrentWeapon == gun)
				Reload();
			else if (PlayerCurrentWeapon == axe)
				PlayerCurrentWeapon = gun;
		}
	}
}

void APlayerCharacter::Dodge()
{
	if (bCanDodge)
	{
		bCanDodge = false;

		FVector DodgeVector = -GetActorForwardVector();

		GetCharacterMovement()->AddImpulse(DodgeVector * DodgeImpulse * GetWorld()->DeltaTimeSeconds,true);

		GetWorldTimerManager().SetTimer(DodgeDelayTimerHandle, this, &APlayerCharacter::ResetDodge, DodgeDelay, false);
	}
}

void APlayerCharacter::Interact()
{
	if (bCanInteract_Axe && AxeInteractionReference != nullptr)
	{
		AxeAmmoInMag++;
		AxeInteractionReference->Destroy();
		AxeInteractionReference = nullptr;
		bCanInteract_Axe = false;
	}
}

void APlayerCharacter::Reload()
{
	if (PlayerCurrentWeapon == gun)
	{
		if (PrimaryAmmoInMag != AmmoMax[0])
		{
			for (int i = 0; i < 40; i++)
			{
				bCanAttack = false;

				if (PrimaryAmmoInMag == AmmoMax[0])
					break;

				if (PrimaryAmmoStored > 0)
				{
					PrimaryAmmoInMag++;
					PrimaryAmmoStored--;
				}
				else
					break;
			}
			EWhenAttacks();
			GetWorldTimerManager().SetTimer(AttackDelayTimerHandle, this, &APlayerCharacter::ResetAttack, 1, false);
		}
	}
}

/* Combat take damage and heal */
void APlayerCharacter::TakeDamage(float Damage, AActor* OtherActor)
{
	if (bCanTakeDamage)
	{
		bCanTakeDamage = false;

		PlayerHealth -= Damage;
		EWhenTakesDamage();

		FVector KnockbackVector = OtherActor->GetActorForwardVector();

		GetCharacterMovement()->AddImpulse(KnockbackVector * DodgeImpulse * GetWorld()->DeltaTimeSeconds, true);
		
		GetWorldTimerManager().SetTimer(TakeDamageDelayTimerHandle, this, &APlayerCharacter::ResetTakeDamage, TakeDamageDelay, false);
		
	}
}

void APlayerCharacter::Heal(float HealAmmount)
{
	PlayerHealth += HealAmmount;
	if (PlayerHealth > MaxPlayerHealth)
	{
		PlayerHealth = MaxPlayerHealth;
	}
	EWhenTakesDamage();
}


void APlayerCharacter::UpdateRotationTowardsCursor(USceneComponent* Comp, float DeltaTime)
{
	FVector MouseDir = FVector::ZeroVector;
	FVector MousePos = FVector::ZeroVector;

	if (APlayerController* PC = Cast<APlayerController>(GetController()))
	{
		PC->DeprojectMousePositionToWorld(MousePos, MouseDir);

		FVector Intersection = FVector::ZeroVector;
		float t = 0.f;

		FVector LineEnd = MousePos + MouseDir * 2000.f;

		bool bIntersectionSuccess = UKismetMathLibrary::LinePlaneIntersection_OriginNormal(
			MousePos,
			LineEnd,
			GetActorLocation(),
			GetActorUpVector(),
			t,
			Intersection
		);
		// Do stuff if line intersected.
		if (bIntersectionSuccess)
		{

			// Calculate direction vector from the pawn body forward vector to intersection vector.
			FVector DirToIntersection = (Intersection - GetActorLocation()).GetSafeNormal();
			// Gets the cosine of the angle between the pawns body forward vector and the direction to intersection.
			float dotForward = Comp->GetForwardVector() | DirToIntersection;
			float Angle = acos(dotForward) * (180.f / PI);
			// Only update the rotation if it is greater to avoid unwanted behaviour.
			if (Angle > .2f)
			{
				// Clamp to limit how fast the component can rotate.
				Angle = FMath::Clamp(Angle, 0.f, RotationMaxSpeed);
				// Gets the cosine of the angle with the right vector against direction to intersection to know on what side of the component is the intersection.
				float dotSide = Comp->GetRightVector() | DirToIntersection;
				// Negates the value depending on what side is the intersection relative to the component.
				Angle *= RotationEase * ((dotSide > 0.f) ? 1.f : -1.f);
				// Create rotator with variable.
				FRotator BodyRotator = FRotator(0.f, Angle * DeltaTime, 0.f);
				// Add rotation to pawn body component.
				Comp->AddRelativeRotation(BodyRotator);
			}
			// Debug
			DrawDebugLine(GetWorld(), GetActorLocation(), Intersection, FColor::Orange, false, -1.f, 0, 4.f);
			DrawDebugSphere(GetWorld(), Intersection, 10.f, 16, FColor::Red, false);
		}
	}
}


/* End Game */
void APlayerCharacter::GameOver()
{
	bCanMove = false;

	CapsuleComponent->Deactivate();
	CapsuleComponent->SetVisibility(false);

	FTimerHandle GameOverTimerHandle;
	GetWorldTimerManager().SetTimer(GameOverTimerHandle, this, &APlayerCharacter::RestartGame, 1.f, false);
}

void APlayerCharacter::RestartGame()
{
	UGameplayStatics::OpenLevel(this, FName(*GetWorld()->GetName()));
}

void APlayerCharacter::AttackAlternative()
{

}
void APlayerCharacter::ItemSlot1()
{
	PlayerCurrentWeapon = gun;
}
void APlayerCharacter::ItemSlot2()
{
	PlayerCurrentWeapon = axe;
}
void APlayerCharacter::RotateCamera_Left() 
{
	FRotator LeftRotation = FRotator(0, 0.7, 0);
	CameraBoom->AddWorldRotation(LeftRotation);
}
void APlayerCharacter::RotateCamera_Right()
{
	FRotator RightRotation = FRotator(0, -0.7, 0);
	CameraBoom->AddWorldRotation(RightRotation);
}


/* Timers end reset */
void APlayerCharacter::ResetAttack()
{
	bCanAttack = true;
	GetWorldTimerManager().ClearTimer(AttackDelayTimerHandle);
}
void APlayerCharacter::ResetDodge()
{
	bCanDodge = true;
	GetWorldTimerManager().ClearTimer(DodgeDelayTimerHandle);
}
void APlayerCharacter::ResetTakeDamage()
{
	bCanTakeDamage = true;
	GetWorldTimerManager().ClearTimer(TakeDamageDelayTimerHandle);
}


void APlayerCharacter::SetInteractVisibility(bool Visible)
{
	if (Visible)
	{
		UI_Interact->SetVisibility(ESlateVisibility::Visible);
		
	}
	else
	{
		UI_Interact->SetVisibility(ESlateVisibility::Hidden);
		
	}
}

/* Hit and Overlap events */
void APlayerCharacter::OnOverlapBegin(UPrimitiveComponent* OverlappedComp, AActor* OtherActor, UPrimitiveComponent* OtherComp, int32 OtherBodyIndex, bool bFromSweep, const FHitResult& SweepResult)
{
	
	if (OtherActor != nullptr)
	{
		AWeapon* Weapon = Cast<AWeapon>(OtherActor);
		AAmmoCrate* AmmoCrate = Cast<AAmmoCrate>(OtherActor);

		if (Weapon)
		{
			SetInteractVisibility(true);
			bCanInteract_Axe = true;
			AxeInteractionReference = Weapon;
		}

		if (AmmoCrate)
		{
			if (AmmoCrate->bIsPrimary)
			{
				if (PrimaryAmmoStored + 60 <= AmmoMaxStored[0])
					PrimaryAmmoStored += 60;
				else
					PrimaryAmmoStored = AmmoMaxStored[0];
			}
			else if (AmmoCrate->bIsAxeAmmo)
			{
				if (AxeAmmoInMag + 1 <= AmmoMax[1])
					AxeAmmoInMag++;
			}
			AmmoCrate->Destroy();
			EWhenAttacks();
		}
	}
	

}

void APlayerCharacter::OnOverlapEnd(UPrimitiveComponent* OverlappedComp, AActor* OtherActor, UPrimitiveComponent* OtherComp, int32 OtherBodyIndex)
{
	if (OtherActor != nullptr)
	{
		AWeapon* Weapon = Cast<AWeapon>(OtherActor);
		
		if (Weapon)
		{
			SetInteractVisibility(false);
			bCanInteract_Axe = false;
			AxeInteractionReference = nullptr;
		}
		
	}
}

void APlayerCharacter::OnHit(UPrimitiveComponent* HitComponent, AActor* OtherActor, UPrimitiveComponent* OtherComp, FVector NormalImpulse, const FHitResult& Hit)
{
	
	if (OtherActor != nullptr)
	{
		

		//AObstacle* Obstacle = Cast<AObstacle>(OtherActor);

		//if (Obstacle)
		//{

		//}

		

	}
	
}